/*
 * File:   lcd.c
 * Author: witkatz
 * Proc: 16F
 * Compiler: X8 V1.32
 */

#include <xc.h>
#include <stdint.h>
#include "lcd.h"

void StrobeLCD(void){
    E_PIN = 1;
    NOP();
    E_PIN = 0;
}

void Pseudo8BitCmdLCD(uint8_t data){
    DATA_PORT = data;
    StrobeLCD();
    delay5msLCD();
}
void InitLCD(uint8_t lcdtype){
    DATA_PORT &= 0x0F;              // upper nibble is data interface
    TRIS_DATA_PORT &= 0x0F;
    TRIS_RW = 0;                    // All control signals made outputs
    TRIS_RS = 0;
    TRIS_E = 0;
    RW_PIN = 0;                     // R/W = 0 -> write data
    RS_PIN = 0;                     // RS = 0 -> command
    E_PIN = 0;                      // Clock = 0
    
    //*** reset procedure
    delay45msLCD();
    Pseudo8BitCmdLCD(0x30);
    Pseudo8BitCmdLCD(0x30);
    WriteCmdLCD(0x32);

    //*** Set data interface width & lines & font, eg InitLCD(FOUR_BIT & LINES_5X7);
    while(BusyLCD());              // Wait if LCD busy
    WriteCmdLCD(lcdtype);          // Function set cmd

    // Turn the display off then on
    while(BusyLCD());                       // Wait if LCD busy
    WriteCmdLCD(DOFF&CURSOR_OFF&BLINK_OFF); // Display OFF/Blink OFF
    while(BusyLCD());                       // Wait if LCD busy
    WriteCmdLCD(DON&CURSOR_OFF&BLINK_OFF);  // Display ON/Blink ON

    // Clear display
    ClearLCD();

    // Set entry mode inc, no shift
    while(BusyLCD());               // Wait if LCD busy
    WriteCmdLCD(SHIFT_CUR_RIGHT);   // Entry Mode

    // Set DD Ram address to 0
    while(BusyLCD());               // Wait if LCD busy
    SetDDRamAddrLCD(0);             // Set Display data ram address to 0

    return;
}
void ClearLCD(void){
    while(BusyLCD());              // Wait if LCD busy
    WriteCmdLCD(0x01);             // Clear display
}
void WriteCmdLCD(uint8_t cmd){
    TRIS_DATA_PORT &= 0x0f;         // 4-bit upper nibble interface
    DATA_PORT &= 0x0f;    
    DATA_PORT |= cmd & 0xf0;        // upper nibble
    RW_PIN = 0;                     // RW = 0 -> write
    RS_PIN = 0;                     // RS = 0 -> command
    delay1usLCD();
    StrobeLCD();
    DATA_PORT &= 0x0f;
    DATA_PORT |= (cmd<<4) & 0xf0;   // lower nibble
    StrobeLCD();
    TRIS_DATA_PORT |= 0xf0;         // Make data nibble input
    
    return;
}
int8_t BusyLCD(void){
    RW_PIN = 1;                     // RW = 1 -> read
    RS_PIN = 0;                     // RS = 0 -> command
    delay1usLCD();
    E_PIN = 1;                      // Clock in the command
    NOP();
    if(DATA_PORT & 0x80)
    {
            E_PIN = 0;              // reset clock line
            NOP();
            StrobeLCD();            // Clock out other nibble
            RW_PIN = 0;             // RW = 0 -> reset to write
            return 1;               // Return TRUE
    }
    else                            // Busy bit is low
    {
            E_PIN = 0;              // reset clock line
            NOP();
            StrobeLCD();            // Clock out other nibble
            RW_PIN = 0;             // RW = 0 -> reset to write
            return 0;               // Return FALSE
    }
}
void SetDDRamAddrLCD(uint8_t DDaddr){
    TRIS_DATA_PORT &= 0x0f;                 // Make port output
    DATA_PORT &= 0x0f;                      // and write upper nibble
    DATA_PORT |= ((DDaddr | 0b10000000) & 0xf0); // upper nibble
    RW_PIN = 0;                             // RW = 0 -> write
    RS_PIN = 0;                             // RS = 0 -> command
    StrobeLCD();                            // Clock the cmd
    DATA_PORT &= 0x0f;                      // Write lower nibble
    DATA_PORT |= ((DDaddr<<4)&0xf0);        // lower nibble
    StrobeLCD();                            // Clock the cmd
    TRIS_DATA_PORT |= 0xf0;                 // Make port input
    return;
}
void WriteCharLCD(char data){
    TRIS_DATA_PORT &= 0x0f;
    DATA_PORT &= 0x0f;
    DATA_PORT |= data & 0xf0;           // upper nibble
    RS_PIN = 1;                         // RS = 1 -> register
    RW_PIN = 0;                         // RW = 0 -> write
    StrobeLCD();                        // Clock nibble into LCD
    DATA_PORT &= 0x0f;
    DATA_PORT |= ((data<<4) & 0xf0);    // lower nibble
    StrobeLCD();                        // Clock nibble into LCD
    TRIS_DATA_PORT |= 0xf0;             // Make port input

    return;
}
void WriteStringLCD(const char *buffer){
    while(*buffer)                  // Write data to LCD up to null
    {
        while(BusyLCD());      // Wait while LCD is busy
        WriteCharLCD(*buffer); // Write character to LCD
        buffer++;               // Increment buffer
    }
    return;
}
char LowerNibbleToHex(uint8_t in){
    char out = (in & 0x0F) + '0';
    if ((in & 0x0F) > 9)
        out += ('A' - '0' - 10);
    return out;
}
void WriteHexLCD(uint8_t data){
    char tmp = LowerNibbleToHex(data >> 4);
    WriteCharLCD(tmp);
    char tmp = LowerNibbleToHex(data);
    WriteCharLCD(tmp);
}
